#!/bin/bash
#
#	Copyright © 2022-2023 Claris International Inc.  All rights reserved.
#

usage()
{
	cat << EOF
Usage: $0 [-h] [-l] [-j] [-s] [-p process] [-a frequency] [-d]
This script gets thread dumps of server processes and Java prorcess of WebDirect.
Available server processes: fmshelper, fmserverd, fmsased, fmsib, fmscwpc, fmwipd, fmodata 
  -h Show this help message.
  -l Get process status only, no trace dumps.
  -j Get thread dump of Java process in WebDirect only.
  -s Get thread dumps of FileMaker Server processes only
  -p Only get thread dumps of specified server processes where names are separated by a colons ':'.
  -a Add a cron job to run the script in the specified 'frequency'. The format is the same used in crontab.
     Frequency fields are: "Minutes", "Hours", "Day of month", "Month", "Day of week", separted by colons ':'.
     Use an asterisk '*' as a wildcard to include all possible values. Remember to quote the string if it contains asterisk characters.
     Use a comma ',' to separate multiple values, e.g. 1,5 means 1 and 5
     Use a hyphen '-' to designate a range of values, e.g. 1-5 means 1, 2, 3, 4, and 5.
     Use a slash '/' to specify step values, e.g. 1-10/2 in the Minutes field means every two minutes in range 1-10, same as 1,3,5,7,9

     * The following command gets only thread dump of WebDirect's Java process.
       GetThreadDumps.sh -j

     * The following command adds a cron job to run the script to get thread dumps of FileMaker Server, Server Assisted Script Engine every
       five minues on week days, but no Java thread dumps.
       GetThreadDumps.sh -s -p fmserverd:fmsased -a "*/5:*:*:*:1-5"

     * The following command removes existing cron job of the script.
       GetThreadDumps.sh -d 

     Note: Since asterisk '*' character are used, it is necessary to quote the frequency string.

  -d Remove the cron job of this script if it exists.
EOF
	exit 0
}

getTimeStamp()
{
	/usr/bin/date "+%m/%d %H:%M:%S"
}

containsElement()
{
#
# returns 0 if there is a match
#
	local result=1
	local toMatch="$1"
	shift
	local e 
	for e; do 
		if [[ "$e" == "$toMatch" ]]; then
			result=0;
			break
		fi
	done
	return "$result"
}

checkPackage()
{
	local result=0
	local cmd
	local pkg
	for cmd in "${!PACKAGE_MAP[@]}"; do
		pkg=${PACKAGE_MAP[$cmd]}
		if [[ ! -f "$cmd" ]]; then
			apt list "$pkg" 2>/dev/null | /usr/bin/grep installed > /dev/null
			local err=$?
			if [[ $err -ne 0 ]]; then
				result=1
				echo "[$(getTimeStamp)] $cmd is needed but its package '$pkg' is not installed." | /usr/bin/tee -a "$LOG_FILE"
				echo "Please run the command 'sudo apt install $pkg' to install..." | /usr/bin/tee -a "$LOG_FILE"
			else 
				result=1
				echo "[$(getTimeStamp)] Package '$pkg' is installed, but '$cmd' is not present." | /usr/bin/tee -a "$LOG_FILE" 
			fi
		fi
	done
	return $result
}

getDeployement()
{
	local enabled
	local comp
	for comp in "${DEPLOYMENT_LIST[@]}"; do
		enabled=$( $TIDY_CMD -xml "$DEPLOYMENT_CONFIG" 2>/dev/null | grep -w "name=\"$comp\"" \
			   | tr " " "\n" | grep -w "enabled=" | cut -d "=" -f2)
		if [ "$enabled" = \"yes\" ]; then
			ENABLED_COMPS+=("$comp")
		fi
	done
}

listCrontab() 
{
	local message=$1
	if [[ -n "$message" ]]; then
		echo "[$(getTimeStamp)] $1 " | /usr/bin/tee -a "$LOG_FILE"
	fi
	/usr/bin/crontab -u 'root' -l 2>/dev/null | /usr/bin/tee -a "$LOG_FILE"
}

getProcs()
{
	local result=$?
	local procState
	local comp
	local name
	echo "------------------------" | /usr/bin/tee -a "$LOG_FILE"
	/usr/bin/ps -ef | head -1 | /usr/bin/tee -a "$LOG_FILE"
	for proc in "${SERVER_PROC_LIST[@]}"; do
		name=${PROC_NAME_MAP["$proc"]}
		case "$proc" in
		java )
			procState=$(/usr/bin/ps -ef | /usr/bin/grep "[j]wpc")
			;;
		node )
			procState=$(/usr/bin/ps -ef | /usr/bin/grep "[n]ode-wip/app")
			;;
		fmshelper | fmserverd | fmsased | fmsib | fmscwpc | fmwipd | fmxdbc_listener | fmodata )
			procState=$(/usr/bin/ps -ef | /usr/bin/grep "$SERVER_DIR/bin/$proc" | /usr/bin/grep -v /usr/bin/grep)
			;;
		esac

		if [[ -n "$procState" ]]; then
#
# Print out process status if it is available
#
			echo "$procState" | /usr/bin/tee -a "$LOG_FILE"
			RUNNING_PROCS+=("$proc")
		else
			comp=${PROC_COMP_MAP["$proc"]}
			containsElement "$comp" "${ENABLED_COMPS[@]}" 
			result=$?
			if [[ $result -eq 0 ]];then
				echo "[ '$name' ($proc) is not running.]"	| /usr/bin/tee -a "$LOG_FILE"
			else
				echo "[ '$name' ($proc) is not enabled.]"	| /usr/bin/tee -a "$LOG_FILE"
			fi						
		fi
	done
	echo  | /usr/bin/tee -a "$LOG_FILE"

	listCrontab "Current crontab of 'root':"
	echo "------------------------" | /usr/bin/tee -a "$LOG_FILE"
	echo  | /usr/bin/tee -a "$LOG_FILE"
	if [[ $GET_PROC_LIST_ONLY = true ]]; then
		exit 0
	fi
}


getServerProcThreadDump()
{
	local result=0
	local enabled
	local logFile
	local cmdFile
	local comp
	local name

	#
	# Loop through running processes to get their thread dumps
	#
	for proc in "${PROC_LIST[@]}"; do
		#
		# Get component name of the process
		#
		if [[ "$proc" == "fmshelper" ]]; then
			enabled=true
		else
			comp=${PROC_COMP_MAP["$proc"]}
			containsElement "$comp" "${ENABLED_COMPS[@]}" 
			result=$?
			if [[ "$result" -eq 0 ]];then
				enabled=true
			else	
				enabled=false
			fi		
		fi	

		#
		# Get component description
		#
		name=${PROC_NAME_MAP["$proc"]}

		#
		# Check if this component is enabled
		#
		if [[ $enabled = "true" ]]; then
			#
			# Java thread dump is done by getJavaThreadDump
			# We do not get thread dump of Node.js
			#
			if [[ $proc = "java" || $proc = "node" ]]; then
				continue
			fi

			procID=$(/usr/bin/pgrep -f "$SERVER_DIR/bin/$proc")
			if [[ -n "$procID" ]]; then
				echo "[$(getTimeStamp)] '$name' ($proc)" | /usr/bin/tee -a "$LOG_FILE"
				logFile=${proc}_$(/usr/bin/date '+%b-%d-%H-%M-%S').log
				cmdFile="/tmp/gdbcmd_${proc}.txt"
		
		#
		# Generate gdb command files on the fly and capture gdb output in /tmp
		#
				cat << EOF > "$cmdFile"
set pagination off	
set confirm off	
attach $procID
set logging off
set print pretty on
set logging file /tmp/$logFile
set logging on
thread apply all backtrace
detach
quit
EOF
				"$GDB_CMD" -x "$cmdFile" > /dev/null 2>&1
				if [[  -f "/tmp/$logFile" ]]; then
					mv -f "/tmp/$logFile" "$LOG_DIR"
					chown "${FILEMAKER_USER}":"${FILEMAKER_GROUP}" "$LOG_DIR/$logFile"
					echo " Thread dump of $proc (PID=$procID) is generated in $LOG_DIR/$logFile" | /usr/bin/tee -a "$LOG_FILE"
				else
					echo " $proc thread dump not generated" | /usr/bin/tee -a "$LOG_FILE"
				fi

				rm -f "$cmdFile"
			else
				echo "[$(getTimeStamp)] '$name' ($proc) is not running" | /usr/bin/tee -a "$LOG_FILE"
			fi
		else
			echo "[$(getTimeStamp)] '$name' ($proc) is not enabled" | /usr/bin/tee -a "$LOG_FILE"
		fi
	
		echo | /usr/bin/tee -a "$LOG_FILE"
	done
		
	chown "${FILEMAKER_USER}":"${FILEMAKER_GROUP}" "$LOG_FILE"
	return "$result"
}

getJavaThreadDump()
{
	local result=0
	local logFile
	local javaPID
	containsElement "jwpc" "${ENABLED_COMPS[@]}" 
	result=$?
	if [[ $result -eq 0 ]]; then
		local comp=${PROC_NAME_MAP["java"]}
		javaPID=$(/usr/bin/ps -ef | /usr/bin/grep '[j]wpc' | awk '{print $2}')
		if [[ -n "$javaPID" ]]; then
			echo "[$(getTimeStamp)] '$comp' (java)" | /usr/bin/tee -a "$LOG_FILE"
			logFile="java_$(/usr/bin/date '+%b-%d-%H-%M-%S').log"
			"$JSTACK_CMD" "$javaPID" > "/tmp/$logFile"
			if [[ -f "/tmp/$logFile" ]]; then
				mv -f "/tmp/$logFile" "$LOG_DIR/$logFile"
				chown "${FILEMAKER_USER}":"${FILEMAKER_GROUP}" "$LOG_DIR/$logFile"
				echo " Thread dump of java (PID=$javaPID) is generated in $LOG_DIR/$logFile" | /usr/bin/tee -a "$JAVA_LOG_FILE"
			else
				result=1
				echo " Java thread dump is not generated." | /usr/bin/tee -a "$JAVA_LOG_FILE"
			fi
		else
			result=1
			echo "[$(getTimeStamp)] '$comp' (java) is not running." | /usr/bin/tee -a "$JAVA_LOG_FILE"
		fi
		echo | /usr/bin/tee -a "$JAVA_LOG_FILE"
		chown "${FILEMAKER_USER}":"${FILEMAKER_GROUP}" "$JAVA_LOG_FILE"
	else
		result=0
	fi

	return "$result"
}


addCronJob()
{
	local result=0
	local frequency=$1
	local cronFields;
	local cronFile="/tmp/crontab.job"

	if [[ -n "$frequency" ]]; then
		cronFields=$(echo "$frequency" | tr ':' ' ')
		if [[ -n $cronFields ]]; then
			echo "[$(getTimeStamp)] Adding '$TOOLS_DIR/GetThreadDumps.sh' to crontab." | /usr/bin/tee -a "$LOG_FILE"
			listCrontab "Current crontab of 'root':"
			#
			# Delete script entry from the crontab and submit a new crontab
			#
			/usr/bin/crontab -u 'root' -l > "$cronFile" 2>/dev/null
			/usr/bin/sed -i -e "/$SCRIPT/d" "$cronFile"
			echo -e "$cronFields\t'$TOOLS_DIR/GetThreadDumps.sh' ${PARAMS[@]}" >> "$cronFile"

			/usr/bin/crontab -u 'root' "$cronFile" 2>&1

			listCrontab "New crontab of 'root':"
			result=$?
		else	
			echo "[$(getTimeStamp)] Cannot set empty field in a cron job." | /usr/bin/tee -a "$LOG_FILE"
			result=1
		fi
	fi

	return "$result"
}

removeCronJob()
{
	local result=0
	local cronFile="/tmp/crontab.job"
	echo "[$(getTimeStamp)] Removing $TOOLS_DIR/GetThreadDumps.sh from crontab." | /usr/bin/tee -a "$LOG_FILE"
	listCrontab "Current crontab of 'root':" 

	#
	# Delete script entry from the crontab and submit a new crontab
	#
	/usr/bin/crontab -u 'root' -l > "$cronFile" 2>/dev/null
	/usr/bin/sed -i -e "/GetThreadDumps.sh/d" "$cronFile"
	/usr/bin/crontab -u 'root' "$cronFile" 2>&1 

	listCrontab "New crontab of 'root':"
	result=$?
	/usr/bin/rm -f "$cronFile"

	exit $result
}

INSTALL_DIR="/opt/FileMaker"
PRODUCT_DIR="$INSTALL_DIR/FileMaker Server"
SERVER_DIR="$PRODUCT_DIR/Database Server"
TOOLS_DIR="$PRODUCT_DIR/Tools"
DEPLOYMENT_CONFIG="$PRODUCT_DIR/Admin/conf/deployment.xml"
LOG_DIR="$PRODUCT_DIR/Logs"
LOG_FILE="$LOG_DIR/serverThreadDumpHistory.log"
JAVA_LOG_FILE="$LOG_DIR/javaThreadDumpHistory.log"
FILEMAKER_USER="fmserver"
FILEMAKER_GROUP="fmsadmin"

GDB_PKG="gdb"
GDB_CMD="/usr/bin/$GDB_PKG"
JDK_VERSION=17
JDK_PKG="openjdk-${JDK_VERSION}-jdk-headless"
JSTACK_CMD="/usr/bin/jstack"
TIDY_PGK="tidy"
TIDY_CMD="/usr/bin/tidy"
RESULT=0
GET_PROC_LIST_ONLY=false
GET_SERVER_STACK=true
GET_JAVA_STACK=true
ADD_CRONJOB=false
DELETE_CRONJOB=false
FREQUENCY=

declare -A PACKAGE_MAP=(
[$GDB_CMD]="$GDB_PKG"
[$JSTACK_CMD]="$JDK_PKG"
[$TIDY_CMD]="$TIDY_PGK"
)

#
# List of server processes we want track
#
declare -a SERVER_PROC_LIST=( fmshelper fmserverd fmsased fmsib fmscwpc fmwipd fmxdbc_listener fmodata node java )

#
# List of server components
#
declare -a DEPLOYMENT_LIST=( server fmse fmsib cwpc wipd xdbc odata wipn jwpc )

#
# A map from process to comonent name 
#
declare -A PROC_COMP_MAP=(
[fmserverd]=server
[fmsased]=fmse
[fmsib]=fmsib
[fmscwpc]=cwpc
[fmxdbc_listener]=xdbc
[fmwipd]=wipd
[fmodata]=odata
[node]=wipn
[java]=jwpc )

#
# A map from process to comonent description 
#
declare -A PROC_NAME_MAP=(
[fmshelper]="FileMaker Server Service" 
[fmserverd]="FileMaker Database Server" 
[fmsased]="FileMaker Script Engine" 
[fmsib]="FileMaker Progressive Backup" 
[fmscwpc]="FileMaker WebDirect Engine" 
[fmxdbc_listener]="FileMaker XDBC Listener"
[fmwipd]="FileMaker Data API Engine"
[fmodata]="FileMaker OData Provider"
[node]="FileMaker Data API Node.js"
[java]="FileMaker WebDirect Java" )

declare -a ENABLED_COMPS

#
# List of running processes, a subset of SERVER_PROC_LIST above
#
declare -a RUNNING_PROCS

#
# List of processes to get thread dumps
#
declare -a PROC_LIST

#
# Command line parameter list
#
declare -a PARAMS

SCRIPT=$(basename "$0")

while getopts "a:dhljsp:" opt
do
	case $opt in 
	a )
		FREQUENCY=$OPTARG
		ADD_CRONJOB=true
		;;
	d )
		DELETE_CRONJOB=true
		;;
	j )
		PARAMS+=("-j")
		GET_SERVER_STACK=false
		;;
	l )
		PARAMS+=("-l")
		GET_PROC_LIST_ONLY=true
		;;
	s )
		PARAMS+=("-s")
		GET_JAVA_STACK=false
		;;
	p )
		PARAMS+=("-p")
		PARAMS+=("$OPTARG")
		PROCS=$OPTARG
		;;
	h )
		usage
		;;
	* )
		usage
		;;
	esac
done

shift $((OPTIND-1))

#
# Make sure we have the privilege to attach to the process
#
if [[ $EUID -ne 0 ]]; then
	echo "You do not have the privilege to run this script. Please run it with 'sudo $0' command."
	exit 1
fi

echo "[$(getTimeStamp)] =============================" >> "$LOG_FILE"

if [[ $DELETE_CRONJOB = true ]]; then
	removeCronJob
fi

checkPackage 
RESULT=$?
if [[ $RESULT -ne 0 ]]; then
	exit $RESULT
fi

getDeployement

getProcs

if [[ $ADD_CRONJOB = true ]]; then
	addCronJob "$FREQUENCY"
fi

if [[ -n $PROCS ]]; then
#
# Parse the process names from command line
#
	IFS=':' read -r -a array <<< "$PROCS"
	for proc in "${array[@]}"; do
		containsElement "$proc" "${SERVER_PROC_LIST[@]}" 
		RESULT=$?
		if [[ "$RESULT" -eq 0 ]]; then
			PROC_LIST+=("$proc")
		else
			echo "Process $proc is not a server process."
		fi
	done
else
#
# Parse the process gathered from default server processes
#
	for proc in "${RUNNING_PROCS[@]}"; do
		PROC_LIST+=("$proc")
	done
fi

if [[ $GET_JAVA_STACK = true ]]; then
	getJavaThreadDump
else
	echo | /usr/bin/tee -a "$LOG_FILE"
fi

if [[ $GET_SERVER_STACK = true ]]; then
	getServerProcThreadDump
else
	echo | /usr/bin/tee -a "$LOG_FILE"
fi


echo "===========================================================================" >> "$LOG_FILE"
echo >> "$LOG_FILE"
